/*******************************************************************************
  ProcessChecker Demo
  Example submitted by David Clegg

  This unit contains the ProcessCheckerSettings class, which is used to store
  the settings for the application.
*******************************************************************************/
using System;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Collections;
using System.Windows.Forms;

namespace ProcessChecker
{

	/// <summary>
	/// Class to persist the settings for ProcessChecker. It is marked with the
	/// [Serializable] attribute so we can serialize it to disk. All other
	/// classes exposed as properties of this class must also be marked as
	/// [Serializable], or have their field declarations marked as
	/// [NonSerialized].
	/// </summary>
	[Serializable]
	public class ProcessCheckerSettings {
		private const int CHECK_FREQUENCY = 5;
		private const bool RESTART_ONE_PROCESS = true;
		private const string SETTINGS_EXTENSION = "dat";

		private ProcessList fWatchedProcesses;
		private int fCheckFrequency;
		private bool fRestartOneProcess;
		private bool fEnabled;

		//I don't want the event handlers to be serialized, but the NonSerialized
		//attribute can only be applied to field declarations. Hence the
		//'non-standard' event declaration.
		[NonSerialized]
		private EventHandler fSettingsLoaded;
		[NonSerialized]
		private EventHandler fSettingsSaved;

		public event EventHandler SettingsLoaded {
			add {fSettingsLoaded += value;}
			remove {fSettingsLoaded -= value;}
		}

		public event EventHandler SettingsSaved {
			add {fSettingsSaved += value;}
			remove {fSettingsSaved -= value;}
		}

		/// <summary>
		/// Container to hold details of all processes to be watched.
		/// </summary>
		public ProcessList WatchedProcesses {
			get {return fWatchedProcesses;}
		}

		/// <summary>
		/// Frequency, in seconds, that we should check to see if any
		/// processes need to be restarted.
		/// </summary>
		public int CheckFrequency {
			get {return fCheckFrequency;}
			set {fCheckFrequency = value;}
		}

		/// <summary>
		/// If this is true, only one process will be restarted every time we
		/// check the watched processes.
		/// </summary>
		public bool RestartOneProcess {
			get {return fRestartOneProcess;}
			set {fRestartOneProcess = value;}
		}

		/// <summary>
		/// Indicates whether process checking is enabled or not.
		/// </summary>
		public bool Enabled {
			get {return fEnabled;}
			set {fEnabled = value;}
		}

		/// <summary>
		/// ProcessCheckerSettings constructor.
		/// </summary>
		public ProcessCheckerSettings() {
			fWatchedProcesses = new ProcessList();
			InitDefaults();
		}

		/// <summary>
		/// Serialize a ProcessCheckerSettings instance from disk.
		/// </summary>
		public static ProcessCheckerSettings Load() {
			ProcessCheckerSettings pcs;
			//The ProcessCheckerSettings will be deserialized from .\ProcessChecker.dat
			string fileName = Path.ChangeExtension(Application.ExecutablePath, SETTINGS_EXTENSION);
			if (File.Exists(fileName)) {
				//Create the FileStream to stream in the .dat file, and
				//deserialize using the BinaryFormatter class.
				FileStream fs = new FileStream(fileName, FileMode.Open);
				try {
					BinaryFormatter bf = new BinaryFormatter();
					pcs = (ProcessCheckerSettings)bf.Deserialize(fs);
				}
				finally {
					fs.Close();
				}
			}
			else {
				//.dat file not found so create a new instance and return that.
				pcs = new ProcessCheckerSettings();
			}
			//Fire the event handler(s) if any delegates have been assigned.
			if (pcs.fSettingsLoaded != null)
				pcs.fSettingsLoaded(pcs, null);
			return pcs;
		}

		/// <summary>
		/// Serialize the ProcessCheckerSettings instance to disk.
		/// </summary>
		public void Save() {
			//The instance will be serialized to .\ProcessChecker.dat
			string fileName = Path.ChangeExtension(Application.ExecutablePath, SETTINGS_EXTENSION);
			FileStream fs = new FileStream(fileName, FileMode.Create);
			try {
				//Create the FileStream to stream out the .dat file, and
				//serialize using the BinaryFormatter class.
				BinaryFormatter bf = new BinaryFormatter();
				bf.Serialize(fs, this);
			}
			finally {
				fs.Close();
			}
			//Fire the event handler(s) if any delegates have been assigned.
			if (fSettingsSaved != null)
				fSettingsSaved(this, null);
		}

		/// <summary>
		/// Initialize default values for a new ProcessCheckerSettings instance.
		/// </summary>
		private void InitDefaults() {
			fCheckFrequency = CHECK_FREQUENCY;
			fRestartOneProcess = RESTART_ONE_PROCESS;
		}
	}
}
